/**
* \file: dbus-interface.c
*
* \version: $Id:$
*
* \release: $Name:$
*
* \component: application-framework
*
* \author: Marko Hoyer / ADIT / SWGII / mhoyer@de.adit-jv.com
*
* \copyright (c) 2016 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
*
***********************************************************************/
#include "dbus_interface.h"

#include "constants.h"
#include "application_state_machine.h"
#include "libtestbedapp_intern.h"

#include "NodeStateTypes.h"
#include "hmi-error-codes.h"
#include "hmi-dbus-constants.h"


#include "hmi_client_dbus_interface.h"
#include "hmi_host_dbus_interface.h"
#include "nsm_consumer.h"
#include "nsm_lifecycle_consumer.h"

//----------------------------------------- private member --------------------------------------------
static gboolean dbus_iface_on_handle_activate (GObject *object, GDBusMethodInvocation *invocation, gpointer user_data);
static gboolean dbus_iface_on_handle_deactivate (GObject *object, GDBusMethodInvocation  *invocation, gpointer user_data);

static void dbus_iface_register_with_NSM(void);
static void dbus_iface_connect_to_HMI(void);
static void dbus_iface_register_with_HMI(void);

static void dbus_iface_on_name_aquired(GDBusConnection *con, const gchar* name, gpointer data);
static void dbus_iface_on_connection_lost(GDBusConnection *con, const gchar* name, gpointer data);

static gboolean dbus_iface_on_lifecycle_request (LifeCycleConsumer *object, GDBusMethodInvocation *invocation,
    guint arg_Request, guint arg_RequestId);
static void dbus_iface_hmihost_name_owner_changed(GObject *object, GParamSpec *pspec, gpointer user_data);

//----------------------------------------- attributes ------------------------------------------------
static const char *cl_busname;

static unsigned int nsm_sd_timeout;

static HMIClient *hmiclient_iface=NULL;
static HMIHost *hmihost_iface=NULL;
static Consumer *nsm_consumer_iface=NULL;
static LifeCycleConsumer *nsm_lifecycle_consumer_iface=NULL;

static guint bus_name_owner_id=0;
//----------------------------------------- public member ---------------------------------------------

tba_error_code_t dbus_iface_init(const char *client_busname, unsigned int shutdown_timeout)
{
	tba_error_code_t result=TBA_RESULT_OK;

	nsm_sd_timeout=shutdown_timeout;

	log_debug("dbus_iface - Initializing dbus interface.");
	cl_busname = client_busname;

	hmiclient_iface=hmiclient_skeleton_new();
	nsm_lifecycle_consumer_iface=life_cycle_consumer_skeleton_new();

	log_debug("asm - Start aquiring bus name: %s", client_busname);
	// the bus is aquired asynchronously, the respective callback triggers the state machine to go on
	bus_name_owner_id = g_bus_own_name(G_BUS_TYPE_SYSTEM, cl_busname,G_BUS_NAME_OWNER_FLAGS_NONE,
									NULL,dbus_iface_on_name_aquired,dbus_iface_on_connection_lost, NULL,NULL);

	
	//expose method calls
	(void)g_signal_connect (hmiclient_iface,"handle_activate",G_CALLBACK (dbus_iface_on_handle_activate),NULL);

	(void)g_signal_connect (hmiclient_iface,"handle_deactivate",G_CALLBACK (dbus_iface_on_handle_deactivate),NULL);
	
	return result;
}


void dbus_iface_emit_state_change_signal(tba_app_state_t new_state)
{
	if (hmiclient_iface!=NULL)
		hmiclient_set_state(hmiclient_iface,(guint)new_state);
}


void dbus_iface_deinit(void)
{
	log_debug("dbus_iface - Deinitializing dbus interface.");
	if (bus_name_owner_id!=0)
	{
			log_debug("dbus_iface - Releasing busname.");
			g_bus_unown_name(bus_name_owner_id);
			bus_name_owner_id=0;
	}

	if (hmiclient_iface!=NULL)
	{
		log_debug("dbus_iface - Destroying HMI client skeleton.");
		/* PRQA: Lint Message 751: gobject type checking at compile time*/
		/*lint -save -e751*/
		g_clear_object(&hmiclient_iface);
		/*lint -restore*/
	}

	if (hmihost_iface!=NULL)
	{
		log_debug("dbus_iface - Destroying HMI host proxy.");
		/* PRQA: Lint Message 751: gobject type checking at compile time*/
		/*lint -save -e751*/
		g_clear_object(&hmihost_iface);
		/*lint -restore*/
	}

	if (nsm_consumer_iface!=NULL)
	{
		log_debug("dbus_iface - Unregistering from Node State Manager.");
		consumer_call_un_register_shutdown_client(nsm_consumer_iface,cl_busname,NSM_LCCONSUMER_OBJECT_PATH,
			NSM_SHUTDOWNTYPE_NORMAL, NULL, NULL,NULL);
		log_debug("dbus_iface - Destroying Node State Manager proxy.");
		/* PRQA: Lint Message 751: gobject type checking at compile time*/
		/*lint -save -e751*/
		g_clear_object(&nsm_consumer_iface);
		/*lint -restore*/
	}

	if (nsm_lifecycle_consumer_iface!=NULL)
	{
		log_debug("dbus_iface - Destroying Lifecycle consumer skeleton.");
		/* PRQA: Lint Message 751: gobject type checking at compile time*/
		/*lint -save -e751*/
		g_clear_object(&nsm_lifecycle_consumer_iface);
		/*lint -restore*/
	}
}

void dbus_iface_answer_pending_lc_request(unsigned int request_id)
{
	GError *err=NULL;
	gint result;

	if (nsm_consumer_iface==NULL)
		return;

	log_debug("dbus_iface - Answering pending lifecycle request (request_id: %d)", request_id);
	consumer_call_lifecycle_request_complete_sync(nsm_consumer_iface, request_id, NsmErrorStatus_Ok,
			&result, NULL,&err);

	if (err!=NULL)
	{
		log_error("Answering pending lifecycle request failed: %s", err->message);
		g_error_free(err);
	}
}

//----------------------------------------- private member --------------------------------------------

static gboolean dbus_iface_on_handle_activate (GObject           *object,
                       GDBusMethodInvocation  *invocation,
                       gpointer                user_data)
{
	log_debug("dbus_iface - method called: Activate()");
	(void)user_data;
	hmiclient_complete_activate((HMIClient *)object,invocation);
	asm_send_to_foreground();
	return TRUE;
} 

static gboolean dbus_iface_on_handle_deactivate (GObject           *object,
                       GDBusMethodInvocation  *invocation,
                       gpointer                user_data)
{
	log_debug("dbus_iface - method called: Deactivate()");
	(void)user_data;
	hmiclient_complete_deactivate((HMIClient *)object,invocation);
	asm_send_to_background();
	return TRUE;
} 

static void dbus_iface_register_with_NSM(void)
{
	gint registration_result=NsmErrorStatus_Ok;
	GError *err=NULL;
	gboolean call_result;

	log_debug("dbus_iface - Creating NSM proxy.");

	//connect with NSM bus
	nsm_consumer_iface=consumer_proxy_new_for_bus_sync(G_BUS_TYPE_SYSTEM, G_DBUS_PROXY_FLAGS_NONE,
			NSM_BUSNAME, NSM_CONSUMER_OBJECT_PATH, NULL, &err);
	if (nsm_consumer_iface==NULL)
	{
		log_info("Creating NSM proxy failed: %s", err->message);
		log_info("Going on anyway.");
		g_error_free(err);
		return;
	}

	log_debug("dbus_iface - Registering as shutdown consumer (Normal shutdown, timeout: %d).", nsm_sd_timeout);
	err=NULL;
	call_result=consumer_call_register_shutdown_client_sync (nsm_consumer_iface,cl_busname,NSM_LCCONSUMER_OBJECT_PATH,
		NSM_SHUTDOWNTYPE_NORMAL, nsm_sd_timeout, &registration_result, NULL, &err);

	if (!call_result ||  registration_result!=NsmErrorStatus_Ok)
	{
		log_info("Registering with Node State Manager as shutdown client failed.");

		if (err!=NULL)
		{
			log_info("Message: %s",err->message);
			/* PRQA: Lint Message 751: gobject type checking at compile time*/
			/*lint -save -e751*/
			g_error_free(err);
			/*lint -restore*/
		}
		else
			log_info("Node State Manager error code: %d",registration_result);


		log_info("Going on anyway.");

		/* PRQA: Lint Message 751: gobject type checking at compile time*/
		/*lint -save -e751*/
		g_clear_object(&nsm_consumer_iface);
		/*lint -restore*/
	}
}

static void dbus_iface_connect_to_HMI(void)
{
	GError *err=NULL;
	log_debug("dbus_iface - Creating HMI host proxy.");
	hmihost_iface=hmihost_proxy_new_for_bus_sync(G_BUS_TYPE_SYSTEM, G_DBUS_PROXY_FLAGS_NONE,
			HMI_BUSNAME, HMI_HOST_OBJECT_PATH, NULL, &err);
	if (hmihost_iface==NULL)
	{
		log_error("Unable to create HMI host proxy: %s", err->message);
		asm_enter_ERROR("Unable to create HMIHost proxy.", TBA_RESULT_DBUS_ERROR);
		g_error_free(err);
		return;
	}

	//register for busname owner changed to get informed when the HMI appears
	(void)g_signal_connect (hmihost_iface,"notify::g-name-owner",G_CALLBACK (dbus_iface_hmihost_name_owner_changed),NULL);

	if(g_dbus_proxy_get_name_owner((GDBusProxy *)hmihost_iface)!=NULL)
	{
		//HMI available
		log_debug("dbus_iface - HMI already available.");
		dbus_iface_register_with_HMI();
	}
	else
		log_debug("dbus_iface - HMI not yet available. Waiting for the HMI to come up.");
}

static void dbus_iface_hmihost_name_owner_changed(GObject *object, GParamSpec *pspec, gpointer user_data)
{
	(void)object;
	(void)pspec;
	(void)user_data;

	if (hmihost_iface==NULL) return;

	if(g_dbus_proxy_get_name_owner((GDBusProxy *)hmihost_iface)!=NULL)
	{
		log_debug("dbus_iface - HMI appeared.");
		//HMI available
		dbus_iface_register_with_HMI();
	}
	else
	{
		//we lost HMI.
		log_info("dbus_iface - HMI disappeared. Waiting for HMI to appear again.");
		asm_hmi_lost_trigger();
	}
}

static void dbus_iface_register_with_HMI(void)
{
	GError *err=NULL;
	guint surface_id;
	guint result;

	asm_set_state(TBA_STATE_HMI_NEGOTIATION);

	log_debug("dbus_iface - Start registering with HMI.");
	if (!hmihost_call_register_application_sync(hmihost_iface,cl_busname,&surface_id, &result,NULL,&err))
	{
		log_error("Failed registering with HMI: %s", err->message);
		asm_enter_ERROR("Unable to register with HMI.", TBA_RESULT_HMI_ERROR);
		g_error_free(err);
		return;
	}

	if (result!=TB_HMI_RESULT_OK)
	{
		switch(result)
		{
			case TB_HMI_RESULT_APP_NOTKNOWN:
				log_error("Configuration issue. HMI does not know about us. Our bus name: %s", cl_busname);
				asm_enter_ERROR("Application not known to the HMI.", TBA_RESULT_HMI_ERROR);
				break;
			case TB_HMI_RESULT_NO_MULTIPLE_APPS:
				log_error("HMI supports only one application at a time and another application is "
						"already registered with the HMI. Our bus name: %s", cl_busname);
				asm_enter_ERROR("HMI supports only one application at a time. Another application "
						"is already registered with the HMI.", TBA_RESULT_HMI_ERROR);
				break;
			default:
				log_error("HMI rejects our registration for an unknown reason. "
						"See HMI logs for further details. Our bus name: %s", cl_busname);
				asm_enter_ERROR("HMI denies registration for an unknown reason.", TBA_RESULT_HMI_ERROR);
				break;
		}
		return;
	}
	asm_hmi_available_trigger(surface_id);
}

static void dbus_iface_on_name_aquired(GDBusConnection *con, const gchar* name, gpointer data)
{
	GError *err=NULL;
	(void)name;
	(void)data;

	log_debug("dbus_iface - Bus name aquired: %s", name);
	g_dbus_connection_set_exit_on_close(con, FALSE);

	(void) g_signal_connect(nsm_lifecycle_consumer_iface, "handle-lifecycle-request", G_CALLBACK(dbus_iface_on_lifecycle_request), NULL);

	/* PRQA: Lint Message 826: deactivation because this is generated code*/
	/*lint -save -e826*/
	if(g_dbus_interface_skeleton_export(G_DBUS_INTERFACE_SKELETON(hmiclient_iface),
                                      con,HMI_CLIENT_OBJECT_PATH,&err) != TRUE)
	{
		log_error("Error exporting HMI client interface: %s",err->message);
		g_error_free(err);
		asm_enter_ERROR("Error exporting the HMI client interface.", TBA_RESULT_DBUS_ERROR);
		return;
	}

	if(g_dbus_interface_skeleton_export(G_DBUS_INTERFACE_SKELETON(nsm_lifecycle_consumer_iface),
                                      con,NSM_LCCONSUMER_OBJECT_PATH,&err) != TRUE)
	{
		log_error("Error exporting Node State Manager client interface: %s", err->message);
		g_error_free(err);
		asm_enter_ERROR("Error exporting the NSM Lifecycle Consumer interface to the system bus.",
				TBA_RESULT_DBUS_ERROR);
		return;
	}
	/*lint -restore*/

	dbus_iface_register_with_NSM();

	dbus_iface_connect_to_HMI();
}

static void dbus_iface_on_connection_lost(GDBusConnection *con, const gchar* name, gpointer data)
{
	(void)con;
	(void)name;
	(void)data;

	log_error("Unable to aquire dbus busname: %s", name);
	asm_enter_ERROR("Unable to aquire dbus busname.", TBA_RESULT_DBUS_ERROR);
}

static gboolean dbus_iface_on_lifecycle_request (LifeCycleConsumer *object, GDBusMethodInvocation *invocation,
    guint arg_Request, guint arg_RequestId)
{
	gboolean transition_finished=true;

	log_debug("dbus_iface - Lifecycle request received from Node State Manager: %u", arg_Request);
	if (arg_Request==NSM_SHUTDOWNTYPE_NORMAL)
		transition_finished=asm_going_to_shutdown_trigger((int)arg_RequestId);
	else if (arg_Request==NSM_SHUTDOWNTYPE_RUNUP)
		transition_finished=asm_waking_up_trigger((int)arg_RequestId);

	log_debug("dbus_iface - Finishing lifecylce request. Response Pending: %s", transition_finished ? "false" : "true");
	if (transition_finished)
		life_cycle_consumer_complete_lifecycle_request(object, invocation,NsmErrorStatus_Ok);
	else
		life_cycle_consumer_complete_lifecycle_request(object, invocation,NsmErrorStatus_ResponsePending);

	return TRUE;
}
